<?php
namespace App\Http\Controllers;

use App\Models\Ticket;
use App\Services\NotificationService;
use Illuminate\Http\Request;

class AdminTicketController extends Controller
{
    private function requireAdmin($user) { if ($user->role !== 'admin') abort(403,'Forbidden'); }

    public function index(Request $request)
    {
        $this->requireAdmin($request->user());
        $q = Ticket::query()->latest();
        if ($request->filled('status')) $q->where('status',$request->string('status'));
        if ($request->filled('priority')) $q->where('priority',$request->string('priority'));
        if ($request->filled('category')) $q->where('category',$request->string('category'));
        return response()->json($q->limit(200)->get());
    }

    public function assign(Request $request, Ticket $ticket)
    {
        $user = $request->user();
        $this->requireAdmin($user);

        $data = $request->validate(['assigned_admin_id'=>'required|integer|exists:users,id']);
        $ticket->assigned_admin_id = $data['assigned_admin_id'];
        if ($ticket->status === 'open') $ticket->status = 'pending';
        $ticket->save();

        $notify = app(NotificationService::class);
        $notify->toUser($ticket->user_id,'تم استلام تذكرتك',"Ticket #{$ticket->id}: {$ticket->subject}",['type'=>'ticket','ticket_id'=>(string)$ticket->id]);

        return response()->json($ticket);
    }

    public function status(Request $request, Ticket $ticket)
    {
        $user = $request->user();
        $this->requireAdmin($user);

        $data = $request->validate([
            'status'=>'required|string|in:open,pending,resolved,closed',
            'priority'=>'nullable|string|in:low,normal,high,urgent',
        ]);

        $ticket->status = $data['status'];
        if (isset($data['priority'])) $ticket->priority = $data['priority'];
        $ticket->save();

        $notify = app(NotificationService::class);
        $notify->toUser($ticket->user_id,'تحديث تذكرتك',"Ticket #{$ticket->id} أصبحت: {$ticket->status}",['type'=>'ticket','ticket_id'=>(string)$ticket->id]);

        return response()->json($ticket);
    }
}
