<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\Vendor;
use App\Models\VendorBranch;
use App\Models\Zone;
use App\Services\DispatchService;
use App\Services\GeoService;
use App\Services\PricingService;
use App\Services\BranchService;
use App\Services\WalletService;
use App\Services\CommissionService;
use App\Services\NotificationService;
use App\Events\OrderStatusUpdated;

class OrderController extends Controller
{
    public function create(Request $request)
    {
        $user = $request->user();
        if ($user->role !== 'customer') return response()->json(['message'=>'Forbidden'], 403);

        $data = $request->validate([
            'vendor_id' => 'required|integer|exists:vendors,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|integer|exists:products,id',
            'items.*.qty' => 'required|integer|min:1',
            'dropoff_address' => 'nullable|string',
            'dropoff_lat' => 'required|numeric|between:-90,90',
            'dropoff_lng' => 'required|numeric|between:-180,180',
            'type' => 'nullable|string|in:delivery,grocery,courier',
            'is_scheduled' => 'nullable|boolean',
            'scheduled_at' => 'nullable|date',
        ]);

        $type = $data['type'] ?? 'delivery';

        return DB::transaction(function () use ($user,$data,$type) {
            $vendor = Vendor::findOrFail($data['vendor_id']);

            // 1) Geofence: find zone hit
            $geo = app(GeoService::class);
            $zones = Zone::where('is_active',true)->get();
            $hitZone = null;
            foreach ($zones as $z) {
                $poly = $z->polygon ?? [];
                if ($poly && $geo->pointInPolygon((float)$data['dropoff_lat'],(float)$data['dropoff_lng'],$poly)) { $hitZone=$z; break; }
            }
            if (!$hitZone) abort(422,'Outside service area');

            // 2) Nearest open branch in same zone
            $branches = VendorBranch::where('vendor_id',$vendor->id)->where('is_active',true)
                ->where('zone_id',$hitZone->id)->with('hours')->get();
            $branchService = app(BranchService::class);
            $pricing = app(PricingService::class);

            $best=null; $bestKm=999999;
            foreach ($branches as $b){
                if(!$branchService->isOpenNow($b)) continue;
                $km = $pricing->haversineKm((float)$b->lat,(float)$b->lng,(float)$data['dropoff_lat'],(float)$data['dropoff_lng']);
                if($km < $bestKm){ $bestKm=$km; $best=$b; }
            }
            if(!$best) abort(422,'No open branch');

            $calc = $pricing->calculate($type, $bestKm);
            if(!$calc['ok']) abort(422, $calc['message']);

            // 3) Subtotal + validate vendor/products
            $productIds = collect($data['items'])->pluck('product_id')->unique()->values();
            $products = Product::whereIn('id',$productIds)->get()->keyBy('id');

            $subtotal = 0;
            foreach ($data['items'] as $it) {
                $p = $products[$it['product_id']] ?? null;
                if (!$p || (int)$p->vendor_id !== (int)$data['vendor_id']) abort(422,'Invalid product/vendor mismatch');
                $subtotal += (float)$p->price * (int)$it['qty'];
            }

            $deliveryFee = (float)$calc['delivery_fee'];
            $total = $subtotal + $deliveryFee;

            $isScheduled = (bool)($data['is_scheduled'] ?? false);
            $scheduledAt = $isScheduled ? ($data['scheduled_at'] ?? null) : null;

            $status = $isScheduled ? 'scheduled' : 'pending';

            $order = Order::create([
                'customer_id'=>$user->id,
                'vendor_id'=>$data['vendor_id'],
                'vendor_branch_id'=>$best->id,
                'type'=>$type,
                'status'=>$status,
                'subtotal'=>$subtotal,
                'delivery_fee'=>$deliveryFee,
                'total'=>$total,
                'distance_km'=>round($bestKm,2),
                'pricing_meta'=>$calc['meta'],
                'dropoff_address'=>$data['dropoff_address'] ?? null,
                'dropoff_lat'=>$data['dropoff_lat'],
                'dropoff_lng'=>$data['dropoff_lng'],
                'is_scheduled'=>$isScheduled,
                'scheduled_at'=>$scheduledAt,
            ]);

            foreach ($data['items'] as $it) {
                $p = $products[$it['product_id']];
                $qty = (int)$it['qty'];
                $unit = (float)$p->price;
                OrderItem::create([
                    'order_id'=>$order->id,
                    'product_id'=>$p->id,
                    'qty'=>$qty,
                    'unit_price'=>$unit,
                    'line_total'=>$unit*$qty,
                ]);
            }

            event(new OrderStatusUpdated($order->id, $order->status));
            return response()->json($order->load('items'), 201);
        });
    }

    public function show(Request $request, Order $order)
    {
        $user = $request->user();
        if ($order->customer_id !== $user->id && $order->driver_id !== $user->id && $user->role !== 'admin') {
            $vendor = Vendor::find($order->vendor_id);
            if (!$vendor || $vendor->owner_user_id !== $user->id) return response()->json(['message'=>'Forbidden'], 403);
        }

        $order->load(['items.product','vendor','branch']);
        return response()->json([
            'id'=>$order->id,
            'status'=>$order->status,
            'total'=>$order->total,
            'delivery_fee'=>$order->delivery_fee,
            'distance_km'=>$order->distance_km,
            'dropoff'=>['lat'=>$order->dropoff_lat,'lng'=>$order->dropoff_lng],
            'branch'=>[
                'lat'=>optional($order->branch)->lat,
                'lng'=>optional($order->branch)->lng,
                'name'=>optional($order->branch)->name,
            ],
            'driver_id'=>$order->driver_id,
        ]);
    }

    public function vendorOrders(Request $request)
    {
        $user = $request->user();
        if ($user->role !== 'vendor') return response()->json(['message'=>'Forbidden'], 403);

        $vendor = Vendor::where('owner_user_id',$user->id)->first();
        if(!$vendor) return response()->json([]);

        return Order::where('vendor_id',$vendor->id)->latest()->with('items')->get();
    }

    public function vendorAccept(Request $request, Order $order, DispatchService $dispatch)
    {
        $user = $request->user();
        if ($user->role !== 'vendor') return response()->json(['message'=>'Forbidden'], 403);

        $vendor = Vendor::find($order->vendor_id);
        if (!$vendor || (int)$vendor->owner_user_id !== (int)$user->id) return response()->json(['message'=>'Forbidden'], 403);

        if (!in_array($order->status, ['pending','paid'], true)) return response()->json(['message'=>'Invalid state'], 422);

        $order->status = 'vendor_accepted';
        $order->save();

        event(new OrderStatusUpdated($order->id, $order->status, ['vendor_id'=>$vendor->id]));

        $driver = $dispatch->assignNearestDriver($order);

        $notify = app(NotificationService::class);
        $notify->toUser($order->customer_id,'تحديث الطلب',"طلب #{$order->id}: {$order->status}",[
            'type'=>'order','order_id'=>(string)$order->id,'status'=>$order->status
        ]);

        return response()->json([
            'order'=>$order->fresh(),
            'driver_assigned'=>(bool)$driver,
            'driver_id'=>$driver?->id,
        ]);
    }

    public function driverJobs(Request $request)
    {
        $user = $request->user();
        if ($user->role !== 'driver') return response()->json(['message'=>'Forbidden'], 403);

        return Order::where('driver_id',$user->id)
            ->whereIn('status',['driver_assigned','driver_accepted','picked_up'])
            ->latest()->with('items')->get();
    }

    public function driverAccept(Request $request, Order $order)
    {
        $user = $request->user();
        if ($user->role !== 'driver') return response()->json(['message'=>'Forbidden'], 403);

        if ((int)$order->driver_id !== (int)$user->id || $order->status !== 'driver_assigned') {
            return response()->json(['message'=>'Invalid state'], 422);
        }

        $order->status = 'driver_accepted';
        $order->save();

        event(new OrderStatusUpdated($order->id, $order->status));
        app(NotificationService::class)->toUser($order->customer_id,'السائق في الطريق',"طلب #{$order->id}: {$order->status}",[
            'type'=>'order','order_id'=>(string)$order->id,'status'=>$order->status
        ]);

        return response()->json($order);
    }

    public function driverDecline(Request $request, Order $order, DispatchService $dispatch)
    {
        $user = $request->user();
        if ($user->role !== 'driver') return response()->json(['message'=>'Forbidden'], 403);

        if ((int)$order->driver_id !== (int)$user->id || $order->status !== 'driver_assigned') {
            return response()->json(['message'=>'Invalid state'], 422);
        }

        $user->driverProfile?->update(['is_available'=>true]);

        $order->driver_id = null;
        $order->status = 'dispatching';
        $order->save();

        event(new OrderStatusUpdated($order->id, $order->status));

        $driver = $dispatch->assignNearestDriver($order);

        return response()->json([
            'order'=>$order->fresh(),
            'reassigned'=>(bool)$driver,
            'driver_id'=>$driver?->id,
        ]);
    }

    public function pickup(Request $request, Order $order)
    {
        $user = $request->user();
        if ($user->role !== 'driver') return response()->json(['message'=>'Forbidden'], 403);

        if ((int)$order->driver_id !== (int)$user->id || $order->status !== 'driver_accepted') {
            return response()->json(['message'=>'Invalid state'], 422);
        }

        $order->otp_code = (string)random_int(1000,9999);
        $order->status = 'picked_up';
        $order->save();

        event(new OrderStatusUpdated($order->id, $order->status));
        app(NotificationService::class)->toUser($order->customer_id,'تم الاستلام',"طلب #{$order->id}: {$order->status}",[
            'type'=>'order','order_id'=>(string)$order->id,'status'=>$order->status
        ]);

        return response()->json(['order'=>$order,'otp_code_for_testing'=>$order->otp_code]);
    }

    public function deliverWithOtp(Request $request, Order $order)
    {
        $user = $request->user();
        if ($user->role !== 'driver') return response()->json(['message'=>'Forbidden'], 403);

        $data = $request->validate(['otp'=>'required|string|min:4|max:6']);

        if ((int)$order->driver_id !== (int)$user->id || $order->status !== 'picked_up') {
            return response()->json(['message'=>'Invalid state'], 422);
        }

        if (!$order->otp_code || $data['otp'] !== $order->otp_code) {
            return response()->json(['message'=>'Invalid OTP'], 422);
        }

        $order->status = 'delivered';
        $order->otp_code = null;
        $order->save();

        // Settlement (vendor + platform + driver) one-time
        if (!$order->is_settled) {
            $wallet = app(WalletService::class);
            $commissionSrv = app(CommissionService::class);

            $segment = 'b2c';
            $percent = $commissionSrv->percentFor($order, $segment);

            $subtotal = (float)$order->subtotal;
            $deliveryFee = (float)$order->delivery_fee;

            $platformCommission = round($subtotal * ($percent/100), 2);
            $vendorNet = round($subtotal - $platformCommission, 2);

            $driverShare = round($deliveryFee * 0.80, 2);
            $platformDeliveryCut = round($deliveryFee - $driverShare, 2);

            $vendor = Vendor::find($order->vendor_id);
            if ($vendor) $wallet->credit($vendor->owner_user_id, $vendorNet, 'vendor_settlement', $order->id, ['percent'=>$percent]);

            if ($order->driver_id) $wallet->credit($order->driver_id, $driverShare, 'driver_earning', $order->id);

            $platformUserId = 1;
            $wallet->credit($platformUserId, $platformCommission, 'platform_commission', $order->id, ['percent'=>$percent]);
            $wallet->credit($platformUserId, $platformDeliveryCut, 'platform_delivery_cut', $order->id);

            $order->is_settled = true;
            $order->save();
        }

        $user->driverProfile?->update(['is_available'=>true]);

        event(new OrderStatusUpdated($order->id, $order->status));
        app(NotificationService::class)->toUser($order->customer_id,'تم التسليم',"طلب #{$order->id}: {$order->status}",[
            'type'=>'order','order_id'=>(string)$order->id,'status'=>$order->status
        ]);

        return response()->json($order);
    }
}
