<?php
namespace App\Services;

use App\Models\DeviceToken;
use Google\Client as GoogleClient;

class NotificationService
{
    public function toUser(int $userId, string $title, string $body, array $data = []): void
    {
        \App\Jobs\SendPushNotification::dispatch($userId, $title, $body, $data)->onQueue('push');
    }

    public function sendNow(int $userId, string $title, string $body, array $data = []): void
    {
        $tokens = DeviceToken::where('user_id', $userId)->pluck('token')->all();
        if (!$tokens) return;

        foreach ($tokens as $token) {
            $this->sendToToken($token, $title, $body, $data);
        }
    }

    private function sendToToken(string $token, string $title, string $body, array $data = []): void
    {
        $projectId = env('FIREBASE_PROJECT_ID');
        if (!$projectId) return;

        $accessToken = $this->getAccessToken();
        if (!$accessToken) return;

        $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

        $dataStr = [];
        foreach ($data as $k => $v) $dataStr[(string)$k] = is_string($v) ? $v : json_encode($v);

        $payload = [
            'message' => [
                'token' => $token,
                'notification' => ['title' => $title,'body' => $body],
                'data' => $dataStr,
            ]
        ];

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer {$accessToken}",
                "Content-Type: application/json",
            ],
            CURLOPT_POSTFIELDS => json_encode($payload),
        ]);
        curl_exec($ch);
        curl_close($ch);
    }

    private function getAccessToken(): ?string
    {
        try {
            $client = new GoogleClient();
            $client->setAuthConfig([
                'type' => 'service_account',
                'project_id' => env('FIREBASE_PROJECT_ID'),
                'client_email' => env('FIREBASE_CLIENT_EMAIL'),
                'private_key' => str_replace("\\n", "\n", env('FIREBASE_PRIVATE_KEY')),
            ]);
            $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
            $client->fetchAccessTokenWithAssertion();
            $token = $client->getAccessToken();
            return $token['access_token'] ?? null;
        } catch (\Throwable $e) {
            return null;
        }
    }
}
