#!/usr/bin/env bash
set -euo pipefail

# =========================
# Platform Installer (Laravel + 3 Next.js PWAs)
# Target: Ubuntu/CentOS servers with Node+npm and Composer installed
# =========================

ROOT_DIR="$(pwd)"
PROJECT_DIR="${ROOT_DIR}/platform"
BACKEND_DIR="${PROJECT_DIR}/backend"
PWA_CUSTOMER_DIR="${PROJECT_DIR}/pwa-customer"
PWA_VENDOR_DIR="${PROJECT_DIR}/pwa-vendor"
PWA_DRIVER_DIR="${PROJECT_DIR}/pwa-driver"

echo "==> Creating project folder: ${PROJECT_DIR}"
mkdir -p "${PROJECT_DIR}"

# ---------- Backend ----------
if [ ! -d "${BACKEND_DIR}" ]; then
  echo "==> Creating Laravel backend (composer create-project)..."
  composer create-project laravel/laravel "${BACKEND_DIR}" --no-interaction
else
  echo "==> Backend directory already exists, skipping create-project."
fi

echo "==> Installing backend deps..."
cd "${BACKEND_DIR}"
composer require laravel/sanctum laravel/reverb google/apiclient --no-interaction
composer require filament/filament:"^3.0" --no-interaction
php artisan vendor:publish --provider="Laravel\\Sanctum\\SanctumServiceProvider" || true
php artisan filament:install || true

echo "==> Applying backend patch..."
rsync -a --delete "${ROOT_DIR}/backend_patch/" "${BACKEND_DIR}/"

echo "==> Backend: running migrations (you must configure .env first if DB not ready)..."
# Attempt migrations; if DB not configured yet, it will fail harmlessly.
php artisan migrate || true

# ---------- PWAs ----------
create_pwa () {
  local name="$1"
  local dir="$2"
  local patch="$3"

  if [ ! -d "${dir}" ]; then
    echo "==> Creating Next.js app: ${name}"
    npx create-next-app@latest "${dir}" --ts --eslint --no-tailwind --no-app-router --use-npm
  else
    echo "==> ${name} already exists, skipping create-next-app."
  fi

  echo "==> Installing ${name} deps..."
  cd "${dir}"
  npm i next-pwa laravel-echo pusher-js @react-google-maps/api firebase

  echo "==> Applying ${name} patch..."
  rsync -a --delete "${patch}/" "${dir}/"
}

cd "${PROJECT_DIR}"
create_pwa "pwa-customer" "${PWA_CUSTOMER_DIR}" "${ROOT_DIR}/pwa-customer_patch"
create_pwa "pwa-vendor"   "${PWA_VENDOR_DIR}"   "${ROOT_DIR}/pwa-vendor_patch"
create_pwa "pwa-driver"   "${PWA_DRIVER_DIR}"   "${ROOT_DIR}/pwa-driver_patch"

echo "==> Done."
echo ""
echo "Next steps:"
echo "1) Configure backend .env at: ${BACKEND_DIR}/.env (DB, APP_URL, MOYASAR, FIREBASE, REVERB)"
echo "2) Run migrations: cd ${BACKEND_DIR} && php artisan migrate"
echo "3) Start services:"
echo "   - Backend API: cd ${BACKEND_DIR} && php artisan serve"
echo "   - Queue:       cd ${BACKEND_DIR} && php artisan queue:work --queue=push,default"
echo "   - Reverb:      cd ${BACKEND_DIR} && php artisan reverb:start"
echo "   - Scheduler:   cd ${BACKEND_DIR} && php artisan schedule:work"
echo "4) Configure each PWA .env.local (API URL, Maps, Reverb, Firebase) then run npm run dev"
