import Link from "next/link";
import { useEffect, useState } from "react";
import { api } from "../src/lib/api";
import { initPushAndGetToken } from "../src/lib/firebase";

type Vendor = { id:number; name:string; type:string };

export default function Home() {
  const [vendors, setVendors] = useState<Vendor[]>([]);
  const [msg, setMsg] = useState<string>("");

  useEffect(() => {
    api<Vendor[]>("/api/vendors").then(setVendors).catch(e=>setMsg(e.message));
  }, []);

  const enablePush = async () => {
    const token = await initPushAndGetToken();
    if (!token) { setMsg("تعذر تفعيل الإشعارات. تأكد من إعداد Firebase ومنح الصلاحية."); return; }
    await api("/api/devices/register", { method:"POST", body: JSON.stringify({ token, platform:"web" }) });
    setMsg("تم تفعيل الإشعارات ✅");
  };

  const Feature = ({ icon, title, desc }: { icon:string; title:string; desc:string }) => (
    <div className="feature" dir="rtl">
      <img src={icon} alt={title} />
      <div>
        <h4>{title}</h4>
        <p>{desc}</p>
      </div>
    </div>
  );

  return (
    <div dir="rtl">
      <div className="nav">
        <div className="container navInner">
          <div className="brand">
            <img src="/assets/logo.svg" alt="Almatajer" style={{ height: 34 }} />
            <span className="pill">جدة • B2C + B2B • تشغيل تجريبي</span>
          </div>
          <div style={{ display:"flex", gap:10, flexWrap:"wrap" }}>
            <button className="btn btnGhost" onClick={enablePush}>تفعيل الإشعارات</button>
            <Link className="btn btnGhost" href="/tickets">الدعم</Link>
            <Link className="btn btnGhost" href="/login">دخول</Link>
            <a className="btn btnGold" href="/vendor" target="_self">بوابة التاجر</a>
            <a className="btn btnGold" href="/driver" target="_self">بوابة السائق</a>
          </div>
        </div>
      </div>

      <section className="hero">
        <div className="container heroGrid">
          <div className="card cardPad">
            <h1 className="h1">منصة المتاجر… تجربة عصريّة فاخرة في تطبيق واحد</h1>
            <div className="sub">
              مطاعم • بقالة • توصيل • صفقات B2B — مع تتبع مباشر وإشعارات فورية ونظام دعم بالتذاكر.
            </div>
            <div className="row">
              <a className="btn btnGold" href="#vendors">ابدأ الطلب الآن</a>
              <a className="btn btnGhost" href="#how">كيف تعمل؟</a>
              <a className="btn btnGhost" href="#features">مزايا المنصة</a>
            </div>

            <div className="kpis" dir="rtl">
              <div className="kpi"><b>Multi‑Vendor</b><span>عدة متاجر في منصة واحدة</span></div>
              <div className="kpi"><b>Live Tracking</b><span>موقع السائق لحظة بلحظة</span></div>
              <div className="kpi"><b>Support Tickets</b><span>نظام تذاكر للدعم والبلاغات</span></div>
            </div>

            {msg && <p style={{ marginTop: 12, color:"#f5d76e" }}>{msg}</p>}
          </div>

          <div className="card">
            <img className="heroImg" src="/assets/hero-1.png" alt="hero" />
          </div>
        </div>
      </section>

      <section id="features" className="section">
        <div className="container">
          <h3 className="sectionTitle">الخدمات الأساسية (MVP)</h3>
          <div className="grid6">
            <Feature icon="/assets/icons/food.svg" title="توصيل المطاعم" desc="سلة، رسوم توصيل حسب المسافة، وتتبع مباشر." />
            <Feature icon="/assets/icons/grocery.svg" title="توصيل البقالة" desc="اختيار متجر، مخزون، طلب سريع وموثوق." />
            <Feature icon="/assets/icons/courier.svg" title="توصيل طرود" desc="طلبات توصيل مرنة، حدود مسافة، وتتبع." />
            <Feature icon="/assets/icons/medicine.svg" title="الأدوية" desc="طلب منتجات صيدلية (حسب تفعيل المتجر)." />
            <Feature icon="/assets/icons/b2b.svg" title="B2B (طلبات كميات)" desc="خيار مخصص للشركات والطلبات الكبيرة." />
            <Feature icon="/assets/icons/support.svg" title="دعم وتذاكر" desc="فتح تذكرة + محادثة + إشعارات." />
          </div>
        </div>
      </section>

      <section id="how" className="section">
        <div className="container split">
          <div className="card cardPad" dir="rtl">
            <h3 className="sectionTitle">كيف تعمل؟</h3>
            <div className="badges">
              <div className="badge"><b>1)</b> اختر متجر/خدمة</div>
              <div className="badge"><b>2)</b> حدد موقع الاستلام/التسليم</div>
              <div className="badge"><b>3)</b> ادفع (ميسر) أو عند الاستلام</div>
              <div className="badge"><b>4)</b> تتبع السائق مباشرة</div>
            </div>
            <p style={{ marginTop: 10, color:"#cbd5e1", lineHeight: 1.7 }}>
              التشغيل مصمم للتوسع تدريجيًا: تبدأ بتجربة محدودة ثم توسع الشبكة حسب الأداء.
            </p>
          </div>
          <div className="card">
            <img className="heroImg" src="/assets/hero-2.png" alt="how" />
          </div>
        </div>
      </section>

      <section id="vendors" className="section">
        <div className="container">
          <h3 className="sectionTitle">المتاجر المتاحة الآن</h3>
          <div style={{ display:"grid", gridTemplateColumns:"repeat(2, minmax(0,1fr))", gap: 12 }}>
            {vendors.map(v => (
              <div key={v.id} className="card cardPad" dir="rtl">
                <div style={{ display:"flex", alignItems:"center", justifyContent:"space-between", gap: 12 }}>
                  <div>
                    <div style={{ fontWeight: 900 }}>{v.name}</div>
                    <div style={{ color:"#cbd5e1", fontSize: 13 }}>تصنيف: {v.type}</div>
                  </div>
                  <Link className="btn btnGold" href={`/vendor/${v.id}`}>عرض المنتجات</Link>
                </div>
              </div>
            ))}
          </div>

          <div className="card cardPad" style={{ marginTop: 12 }} dir="rtl">
            <div style={{ display:"flex", alignItems:"center", justifyContent:"space-between", gap: 12, flexWrap:"wrap" }}>
              <div>
                <div style={{ fontWeight: 900, fontSize: 16 }}>هل أنت تاجر وتريد الانضمام؟</div>
                <div style={{ color:"#cbd5e1", fontSize: 13, marginTop: 6 }}>
                  لوحة تاجر + طلبات + إشعارات + تسوية العمولات.
                </div>
              </div>
              <a className="btn btnGhost" href="/vendor" target="_self">فتح بوابة التاجر</a>
            </div>
          </div>
        </div>
      </section>

      <footer className="footer">
        <div className="container" dir="rtl">
          <div style={{ display:"flex", justifyContent:"space-between", gap: 12, flexWrap:"wrap", alignItems:"center" }}>
            <div>
              <div style={{ fontWeight: 900 }}>Almatajer</div>
              <div style={{ fontSize: 13 }}>تشغيل تجريبي — جدة</div>
            </div>
            <div style={{ display:"flex", gap: 10, flexWrap:"wrap" }}>
              <a className="pill" href="/tickets">الدعم</a>
              <a className="pill" href="/vendor/landing">التاجر</a>
              <a className="pill" href="/driver/landing">السائق</a>
            </div>
          </div>
          <div style={{ marginTop: 12, fontSize: 12, opacity: 0.75 }}>
            © {new Date().getFullYear()} Almatajer. جميع الحقوق محفوظة.
          </div>
        </div>
      </footer>
    </div>
  );
}
