import Link from "next/link";
import { useEffect, useState } from "react";
import { api } from "../src/lib/api";
import { startDriverTracking } from "../src/lib/tracking";
import { initPushAndGetToken } from "../src/lib/firebase";

type Order = { id:number; status:string; total:number };

export default function DriverHome() {
  const [orders, setOrders] = useState<Order[]>([]);
  const [msg, setMsg] = useState("");
  const [stopper, setStopper] = useState<null | (()=>void)>(null);

  const load = async () => {
    try { setOrders(await api<Order[]>("/api/driver/jobs")); } catch (e:any) { setMsg(e.message); }
  };

  useEffect(()=>{ load(); }, []);

  const toggleTracking = () => {
    try {
      if (stopper) { stopper(); setStopper(null); setMsg("Tracking stopped"); return; }
      const stop = startDriverTracking();
      setStopper(()=>stop);
      setMsg("Tracking started");
    } catch (e:any) { setMsg(e.message); }
  };

  const accept = async (id:number) => { await api(`/api/driver/jobs/${id}/accept`, { method:"POST" }); load(); };
  const decline = async (id:number) => { await api(`/api/driver/jobs/${id}/decline`, { method:"POST" }); load(); };
  const pickup = async (id:number) => {
    const res = await api<any>(`/api/orders/${id}/pickup`, { method:"POST" });
    setMsg(`Picked up. OTP(test): ${res.otp_code_for_testing}`);
    load();
  };
  const deliver = async (id:number) => {
    const otp = prompt("Enter OTP from customer:");
    if (!otp) return;
    await api(`/api/orders/${id}/deliver`, { method:"POST", body: JSON.stringify({ otp }) });
    setMsg("Delivered ✅");
    load();
  };

  const enablePush = async () => {
    const token = await initPushAndGetToken();
    if (!token) return setMsg("Push token not available. Check Firebase config / permissions.");
    await api("/api/devices/register", { method:"POST", body: JSON.stringify({ token, platform:"web" }) });
    setMsg("Push enabled ✅");
  };

  return (
    <div style={{ maxWidth: 820, margin:"24px auto", padding:16 }}>
      <div style={{ display:"flex", justifyContent:"space-between" }}>
        <h2>Driver</h2>
        <div style={{ display:"flex", gap:12 }}>
          <button onClick={enablePush}>Enable Push</button>
          <Link href="/login">Login</Link>
        </div>
      </div>

      {msg && <p>{msg}</p>}

      <button onClick={toggleTracking} style={{ padding:10, marginBottom:12 }}>
        {stopper ? "Stop Tracking" : "Start Tracking"}
      </button>
      <button onClick={load} style={{ padding:10, marginLeft:8 }}>Refresh</button>

      {orders.map(o => (
        <div key={o.id} style={{ border:"1px solid #ddd", padding:12, marginTop:10, borderRadius:10 }}>
          <div>Order #{o.id} — <b>{o.status}</b> — Total: {o.total}</div>
          <div style={{ marginTop:8, display:"flex", gap:8, flexWrap:"wrap" }}>
            {o.status === "driver_assigned" && (
              <>
                <button onClick={()=>accept(o.id)}>Accept</button>
                <button onClick={()=>decline(o.id)}>Decline</button>
              </>
            )}
            {o.status === "driver_accepted" && <button onClick={()=>pickup(o.id)}>Pickup</button>}
            {o.status === "picked_up" && <button onClick={()=>deliver(o.id)}>Deliver (OTP)</button>}
          </div>
        </div>
      ))}
    </div>
  );
}
